import os
import io
import csv
import json
import time
import secrets
import requests
import telebot
from telebot import types

from db import DB
from security import hash_password, verify_password

def now() -> int:
    return int(time.time())

def make_main_menu() -> types.ReplyKeyboardMarkup:
    kb = types.ReplyKeyboardMarkup(resize_keyboard=True)
    kb.row("📚 دوره‌ها", "👤 اطلاعات کاربری")
    kb.row("🧑‍💻 پشتیبانی")
    kb.row("⚙️ تنظیمات")
    return kb

def make_phone_request_kb() -> types.ReplyKeyboardMarkup:
    kb = types.ReplyKeyboardMarkup(resize_keyboard=True, one_time_keyboard=True)
    kb.row(types.KeyboardButton("📱 ارسال شماره", request_contact=True))
    return kb

def glass_inline(rows):
    kb = types.InlineKeyboardMarkup()
    for row in rows:
        kb.row(*row)
    return kb

class BotApp:
    def __init__(self, bot: telebot.TeleBot, db: DB, base_url: str, data_dir: str):
        self.bot = bot
        self.db = db
        self.base_url = base_url
        self.data_dir = data_dir

        self._ensure_defaults()
        self._register_handlers()

    def _ensure_defaults(self):
        # Password hash default (do not display in bot)
        if not self.db.get_setting("admin_pw_hash"):
            self.db.set_setting("admin_pw_hash", hash_password("8137"))

        if self.db.get_setting("bot_enabled") is None:
            self.db.set_setting("bot_enabled", "1")

        if self.db.get_setting("link_expiry_sec") is None:
            self.db.set_setting("link_expiry_sec", str(10 * 60))

        if self.db.get_setting("msg_caption") is None:
            self.db.set_setting("msg_caption", "✅ مشاهده فقط برای دانش‌آموزان تاییدشده")

        if self.db.get_setting("webapp_text") is None:
            self.db.set_setting("webapp_text", "این محتوا مخصوص دانش‌آموزان دوره است.")

    def _bot_enabled(self) -> bool:
        return self.db.get_setting("bot_enabled", "1") == "1"

    def _link_expiry(self) -> int:
        try:
            return int(self.db.get_setting("link_expiry_sec", str(600)))
        except Exception:
            return 600

    def _required_chat(self):
        chat_id = self.db.get_setting("required_chat_id")
        invite = self.db.get_setting("required_invite_url")
        if not chat_id:
            return None, invite
        try:
            return int(chat_id), invite
        except Exception:
            return None, invite

    def _is_member_required(self, tg_id: int) -> bool:
        required_chat_id, _ = self._required_chat()
        if not required_chat_id:
            return True
        try:
            cm = self.bot.get_chat_member(required_chat_id, tg_id)
            return cm.status in ("member", "administrator", "creator")
        except Exception:
            # If bot not admin or no access, be strict
            return False

    def _is_admin(self, tg_id: int) -> bool:
        return self.db.is_admin(tg_id)

    def _admin_authed(self, tg_id: int) -> bool:
        # Simple auth session: stored in user.state == ADMIN_OK with tmp expiry
        u = self.db.get_user(tg_id)
        if not u:
            return False
        if u["state"] != "ADMIN_OK":
            return False
        tmp = {}
        try:
            tmp = json.loads(u["tmp_json"] or "{}")
        except Exception:
            tmp = {}
        exp = int(tmp.get("admin_exp", 0) or 0)
        return exp > now()

    def _set_admin_ok(self, tg_id: int, minutes: int = 30):
        self.db.set_user_state(tg_id, "ADMIN_OK", {"admin_exp": now() + minutes * 60})

    def _require_admin(self, message) -> bool:
        tg_id = message.from_user.id
        if not self._is_admin(tg_id):
            self.bot.send_message(message.chat.id, "⛔️ دسترسی ندارید.", reply_markup=make_main_menu())
            return False
        if not self._admin_authed(tg_id):
            self.db.set_user_state(tg_id, "ADMIN_AWAIT_PW", {})
            self.bot.send_message(message.chat.id, "🔐 رمز ورود تنظیمات را وارد کنید:", reply_markup=types.ReplyKeyboardRemove())
            return False
        return True

    def _require_student(self, chat_id: int, tg_id: int) -> bool:
        if not self._bot_enabled() and not self._is_admin(tg_id):
            self.bot.send_message(chat_id, "⛔️ بات موقتاً خاموش است.", reply_markup=make_main_menu())
            return False

        if not self._is_member_required(tg_id):
            _, invite = self._required_chat()
            txt = "⛔️ برای استفاده، عضویت اجباری فعال است.\n"
            if invite:
                txt += f"🔗 لینک عضویت: {invite}\n"
            txt += "بعد از عضویت دوباره /start را بزنید."
            self.bot.send_message(chat_id, txt, reply_markup=types.ReplyKeyboardRemove())
            return False

        u = self.db.get_user(tg_id)
        if not u or not u["phone"]:
            self.db.set_user_state(tg_id, "AWAIT_PHONE", {})
            self.bot.send_message(chat_id, "📱 برای اولین ورود، شماره‌تان را ارسال کنید.", reply_markup=make_phone_request_kb())
            return False

        if int(u["is_verified"]) != 1:
            support = self.db.get_setting("support_username") or ""
            msg = "⛔️ شما به عنوان دانش‌آموز تایید نشده‌اید.\n"
            msg += "اگر دانش‌آموز هستید، از بخش 🧑‍💻 پشتیبانی پیام بدهید."
            if support:
                msg += f"\nپشتیبان: @{support.lstrip('@')}"
            self.bot.send_message(chat_id, msg, reply_markup=make_main_menu())
            return False

        return True

    def _register_handlers(self):
        bot = self.bot

        @bot.message_handler(commands=["start"])
        def start(m: types.Message):
            self.db.upsert_user_seen(m.from_user.id, m.from_user.username)
            # show menu; access enforced when entering restricted sections
            bot.send_message(m.chat.id, "سلام!\nاز منو انتخاب کنید.", reply_markup=make_main_menu())

        @bot.message_handler(content_types=["contact"])
        def on_contact(m: types.Message):
            self.db.upsert_user_seen(m.from_user.id, m.from_user.username)
            if not m.contact or not m.contact.phone_number:
                bot.send_message(m.chat.id, "⚠️ شماره معتبر نبود.", reply_markup=make_phone_request_kb())
                return
            # Only accept user's own contact
            if m.contact.user_id and m.contact.user_id != m.from_user.id:
                bot.send_message(m.chat.id, "⛔️ فقط شماره خودتان را ارسال کنید.", reply_markup=make_phone_request_kb())
                return
            phone = m.contact.phone_number
            self.db.set_user_phone(m.from_user.id, phone)
            self.db.set_user_state(m.from_user.id, None, {})
            bot.send_message(m.chat.id, "✅ شماره ثبت شد.", reply_markup=make_main_menu())

        @bot.message_handler(func=lambda msg: True, content_types=["text"])
        def on_text(m: types.Message):
            self.db.upsert_user_seen(m.from_user.id, m.from_user.username)
            u = self.db.get_user(m.from_user.id)
            state = (u["state"] if u else None) or ""

            # Admin password input
            if state == "ADMIN_AWAIT_PW":
                pw_hash = self.db.get_setting("admin_pw_hash", "")
                if pw_hash and verify_password(m.text.strip(), pw_hash):
                    self._set_admin_ok(m.from_user.id, minutes=30)
                    bot.send_message(m.chat.id, "✅ وارد تنظیمات شدید.", reply_markup=self._admin_menu())
                else:
                    bot.send_message(m.chat.id, "⛔️ رمز اشتباه است.", reply_markup=make_main_menu())
                    self.db.set_user_state(m.from_user.id, None, {})
                return

            # Admin flows
            if state == "ADMIN_SET_PW":
                newpw = m.text.strip()
                if len(newpw) < 4:
                    bot.send_message(m.chat.id, "⚠️ رمز خیلی کوتاه است. دوباره بفرستید:")
                    return
                self.db.set_setting("admin_pw_hash", hash_password(newpw))
                self.db.set_user_state(m.from_user.id, "ADMIN_OK", {"admin_exp": now() + 30*60})
                bot.send_message(m.chat.id, "✅ رمز تغییر کرد.", reply_markup=self._admin_menu())
                return

            if state == "ADMIN_ADD_ADMIN":
                try:
                    target = int(m.text.strip())
                except Exception:
                    bot.send_message(m.chat.id, "⚠️ فقط آیدی عددی بفرستید.")
                    return
                self.db.add_admin(target)
                self.db.set_user_state(m.from_user.id, "ADMIN_OK", {"admin_exp": now() + 30*60})
                bot.send_message(m.chat.id, f"✅ ادمین اضافه شد: {target}", reply_markup=self._admin_menu())
                return

            if state == "ADMIN_SET_SUPPORT":
                username = m.text.strip().lstrip("@")
                if not username:
                    bot.send_message(m.chat.id, "⚠️ نام کاربری معتبر نیست.")
                    return
                self.db.set_setting("support_username", username)
                self.db.set_user_state(m.from_user.id, "ADMIN_OK", {"admin_exp": now() + 30*60})
                bot.send_message(m.chat.id, f"✅ پشتیبان تنظیم شد: @{username}", reply_markup=self._admin_menu())
                return

            if state == "ADMIN_SET_LINKEXP":
                try:
                    minutes = int(m.text.strip())
                    if minutes < 1 or minutes > 24*60:
                        raise ValueError()
                except Exception:
                    bot.send_message(m.chat.id, "⚠️ عدد دقیقه (بین 1 تا 1440) بفرستید.")
                    return
                self.db.set_setting("link_expiry_sec", str(minutes*60))
                self.db.set_user_state(m.from_user.id, "ADMIN_OK", {"admin_exp": now() + 30*60})
                bot.send_message(m.chat.id, f"✅ زمان لینک تنظیم شد: {minutes} دقیقه", reply_markup=self._admin_menu())
                return

            if state == "ADMIN_SET_REQUIRED_CHAT":
                # format: chat_id | invite_url(optional)
                parts = [p.strip() for p in m.text.split() if p.strip()]
                if not parts:
                    bot.send_message(m.chat.id, "⚠️ مقدار صحیح نیست.")
                    return
                try:
                    chat_id = int(parts[0])
                except Exception:
                    bot.send_message(m.chat.id, "⚠️ chat_id باید عدد باشد (مثلاً -100123...).")
                    return
                invite = parts[1] if len(parts) > 1 else ""
                self.db.set_setting("required_chat_id", str(chat_id))
                self.db.set_setting("required_invite_url", invite)
                self.db.set_user_state(m.from_user.id, "ADMIN_OK", {"admin_exp": now() + 30*60})
                bot.send_message(m.chat.id, "✅ عضویت اجباری تنظیم شد.", reply_markup=self._admin_menu())
                return

            if state == "ADMIN_SET_WEBAPP_TEXT":
                self.db.set_setting("webapp_text", m.text)
                self.db.set_user_state(m.from_user.id, "ADMIN_OK", {"admin_exp": now() + 30*60})
                bot.send_message(m.chat.id, "✅ متن WebApp تنظیم شد.", reply_markup=self._admin_menu())
                return

            if state == "ADMIN_SET_CAPTION":
                self.db.set_setting("msg_caption", m.text)
                self.db.set_user_state(m.from_user.id, "ADMIN_OK", {"admin_exp": now() + 30*60})
                bot.send_message(m.chat.id, "✅ متن ضمیمه پیام‌ها تنظیم شد.", reply_markup=self._admin_menu())
                return

            if state == "ADMIN_NEW_COURSE_TITLE":
                title = m.text.strip()
                self.db.set_user_state(m.from_user.id, "ADMIN_NEW_COURSE_DESC", {"title": title})
                bot.send_message(m.chat.id, "📝 توضیح دوره را بفرستید (یا یک خط کوتاه):")
                return

            if state == "ADMIN_NEW_COURSE_DESC":
                tmp = {}
                try:
                    tmp = json.loads(u["tmp_json"] or "{}")
                except Exception:
                    tmp = {}
                title = tmp.get("title", "بدون عنوان")
                desc = m.text.strip()
                cid = self.db.create_course(title, desc)
                self.db.set_user_state(m.from_user.id, "ADMIN_OK", {"admin_exp": now() + 30*60})
                bot.send_message(m.chat.id, f"✅ دوره ساخته شد. ID: {cid}", reply_markup=self._admin_menu())
                return

            if state == "ADMIN_UPLOAD_LESSON_TITLE":
                tmp = {}
                try:
                    tmp = json.loads(u["tmp_json"] or "{}")
                except Exception:
                    tmp = {}
                course_id = int(tmp.get("course_id", 0) or 0)
                if not course_id:
                    self.db.set_user_state(m.from_user.id, "ADMIN_OK", {"admin_exp": now() + 30*60})
                    bot.send_message(m.chat.id, "⚠️ خطا: دوره نامشخص.", reply_markup=self._admin_menu())
                    return
                lesson_title = m.text.strip()
                self.db.set_user_state(m.from_user.id, "ADMIN_UPLOAD_WAIT_VIDEO", {"course_id": course_id, "lesson_title": lesson_title})
                bot.send_message(m.chat.id, "🎬 حالا ویدیو را ارسال کنید (Video یا File).")
                return

            # User profile submit
            if state == "USER_AWAIT_PROFILE":
                text = m.text.strip()
                self.db.set_user_profile_text(m.from_user.id, text)
                self.db.set_user_state(m.from_user.id, None, {})
                # notify admins
                self._notify_admins_profile(m.from_user.id)
                bot.send_message(m.chat.id, "✅ اطلاعات شما ارسال شد. بعد از تایید دسترسی فعال می‌شود.", reply_markup=make_main_menu())
                return

            # Main menu clicks
            if m.text == "⚙️ تنظیمات":
                if self._is_admin(m.from_user.id):
                    if self._admin_authed(m.from_user.id):
                        bot.send_message(m.chat.id, "⚙️ پنل مدیریت", reply_markup=self._admin_menu())
                    else:
                        self.db.set_user_state(m.from_user.id, "ADMIN_AWAIT_PW", {})
                        bot.send_message(m.chat.id, "🔐 رمز ورود تنظیمات را وارد کنید:", reply_markup=types.ReplyKeyboardRemove())
                else:
                    bot.send_message(m.chat.id, "⛔️ دسترسی ندارید.", reply_markup=make_main_menu())
                return

            if m.text == "👤 اطلاعات کاربری":
                # require phone first
                uu = self.db.get_user(m.from_user.id)
                if not uu or not uu["phone"]:
                    self.db.set_user_state(m.from_user.id, "AWAIT_PHONE", {})
                    bot.send_message(m.chat.id, "📱 ابتدا شماره را ارسال کنید.", reply_markup=make_phone_request_kb())
                    return
                self.db.set_user_state(m.from_user.id, "USER_AWAIT_PROFILE", {})
                bot.send_message(
                    m.chat.id,
                    "🧾 اطلاعاتتان را یکجا ارسال کنید.\n"
                    "مثال:\n"
                    "نام و نام خانوادگی:\n"
                    "نام دوره:\n"
                    "شماره سفارش/رسید (اگر دارید):",
                    reply_markup=types.ReplyKeyboardRemove(),
                )
                return

            if m.text == "🧑‍💻 پشتیبانی":
                support = self.db.get_setting("support_username") or ""
                if support:
                    link = f"https://t.me/{support.lstrip('@')}"
                    bot.send_message(m.chat.id, f"🧑‍💻 پشتیبانی: {link}", reply_markup=make_main_menu())
                else:
                    bot.send_message(m.chat.id, "⚠️ هنوز پشتیبان تنظیم نشده.", reply_markup=make_main_menu())
                return

            if m.text == "📚 دوره‌ها":
                if not self._require_student(m.chat.id, m.from_user.id):
                    return
                self._show_courses(m.chat.id, m.from_user.id)
                return

            # default
            bot.send_message(m.chat.id, "از منو انتخاب کنید.", reply_markup=make_main_menu())

        @bot.message_handler(content_types=["video", "document"])
        def on_video_or_doc(m: types.Message):
            self.db.upsert_user_seen(m.from_user.id, m.from_user.username)
            u = self.db.get_user(m.from_user.id)
            state = (u["state"] if u else None) or ""
            if state != "ADMIN_UPLOAD_WAIT_VIDEO":
                return
            if not self._require_admin(m):
                return

            tmp = {}
            try:
                tmp = json.loads(u["tmp_json"] or "{}")
            except Exception:
                tmp = {}
            course_id = int(tmp.get("course_id", 0) or 0)
            lesson_title = tmp.get("lesson_title", "درس جدید")

            # Extract file_id
            file_id = None
            if m.content_type == "video" and m.video:
                file_id = m.video.file_id
            elif m.content_type == "document" and m.document:
                file_id = m.document.file_id

            if not file_id:
                self.bot.send_message(m.chat.id, "⚠️ فایل معتبر نبود.")
                return

            # Create lesson + upload record
            lesson_id = self.db.create_lesson(course_id, lesson_title)

            # download original from Telegram
            original_dir = os.path.join(self.data_dir, "originals", str(lesson_id))
            os.makedirs(original_dir, exist_ok=True)
            original_path = os.path.join(original_dir, "original.mp4")

            try:
                self._download_telegram_file(file_id, original_path)
            except Exception as e:
                self.db.set_lesson_status(lesson_id, "error")
                self.bot.send_message(m.chat.id, f"⛔️ خطا در دانلود فایل از تلگرام: {e}")
                self.db.set_user_state(m.from_user.id, "ADMIN_OK", {"admin_exp": now() + 30*60})
                return

            upload_id = self.db.create_upload(course_id, lesson_id, file_id, original_path)
            self.db.set_user_state(m.from_user.id, "ADMIN_OK", {"admin_exp": now() + 30*60})
            self.bot.send_message(
                m.chat.id,
                f"✅ ویدیو دریافت شد و برای تبدیل به کیفیت‌های مختلف در صف قرار گرفت.\n"
                f"ID آپلود: {upload_id}\n"
                f"درس: {lesson_title}",
                reply_markup=self._admin_menu()
            )

        @bot.callback_query_handler(func=lambda call: True)
        def on_callback(call: types.CallbackQuery):
            self.db.upsert_user_seen(call.from_user.id, call.from_user.username)
            data = call.data or ""

            # USER: course / lesson / quality
            if data.startswith("C_"):
                if not self._require_student(call.message.chat.id, call.from_user.id):
                    bot.answer_callback_query(call.id)
                    return
                course_id = int(data.split("_", 1)[1])
                self._show_lessons(call.message.chat.id, call.from_user.id, course_id)
                bot.answer_callback_query(call.id)
                return

            if data.startswith("L_"):
                if not self._require_student(call.message.chat.id, call.from_user.id):
                    bot.answer_callback_query(call.id)
                    return
                lesson_id = int(data.split("_", 1)[1])
                self._show_qualities(call.message.chat.id, call.from_user.id, lesson_id)
                bot.answer_callback_query(call.id)
                return

            if data.startswith("Q_"):
                if not self._require_student(call.message.chat.id, call.from_user.id):
                    bot.answer_callback_query(call.id)
                    return
                _, media_id_s = data.split("_", 1)
                media_id = int(media_id_s)
                self._send_webapp_link(call.message.chat.id, call.from_user.id, media_id)
                bot.answer_callback_query(call.id)
                return

            # ADMIN
            if data == "A_MENU":
                if not self._require_admin(call.message):
                    bot.answer_callback_query(call.id)
                    return
                bot.edit_message_text("⚙️ پنل مدیریت", call.message.chat.id, call.message.message_id, reply_markup=self._admin_menu_inline())
                bot.answer_callback_query(call.id)
                return

            if data == "A_TOGGLE_BOT":
                if not self._require_admin(call.message):
                    bot.answer_callback_query(call.id); return
                cur = self.db.get_setting("bot_enabled", "1")
                self.db.set_setting("bot_enabled", "0" if cur == "1" else "1")
                bot.answer_callback_query(call.id, "انجام شد")
                bot.edit_message_reply_markup(call.message.chat.id, call.message.message_id, reply_markup=self._admin_menu_inline())
                return

            if data == "A_SET_PW":
                if not self._require_admin(call.message):
                    bot.answer_callback_query(call.id); return
                self.db.set_user_state(call.from_user.id, "ADMIN_SET_PW", {})
                bot.send_message(call.message.chat.id, "🔐 رمز جدید را ارسال کنید:", reply_markup=types.ReplyKeyboardRemove())
                bot.answer_callback_query(call.id)
                return

            if data == "A_ADD_ADMIN":
                if not self._require_admin(call.message):
                    bot.answer_callback_query(call.id); return
                self.db.set_user_state(call.from_user.id, "ADMIN_ADD_ADMIN", {})
                bot.send_message(call.message.chat.id, "👮 آیدی عددی ادمین جدید را بفرستید:", reply_markup=types.ReplyKeyboardRemove())
                bot.answer_callback_query(call.id)
                return

            if data == "A_SET_SUPPORT":
                if not self._require_admin(call.message):
                    bot.answer_callback_query(call.id); return
                self.db.set_user_state(call.from_user.id, "ADMIN_SET_SUPPORT", {})
                bot.send_message(call.message.chat.id, "🧑‍💻 نام کاربری پشتیبان را بفرستید (مثلاً support_username):", reply_markup=types.ReplyKeyboardRemove())
                bot.answer_callback_query(call.id)
                return

            if data == "A_SET_LINKEXP":
                if not self._require_admin(call.message):
                    bot.answer_callback_query(call.id); return
                self.db.set_user_state(call.from_user.id, "ADMIN_SET_LINKEXP", {})
                bot.send_message(call.message.chat.id, "⏳ زمان انقضای لینک (دقیقه) را بفرستید:", reply_markup=types.ReplyKeyboardRemove())
                bot.answer_callback_query(call.id)
                return

            if data == "A_SET_REQ":
                if not self._require_admin(call.message):
                    bot.answer_callback_query(call.id); return
                self.db.set_user_state(call.from_user.id, "ADMIN_SET_REQUIRED_CHAT", {})
                bot.send_message(
                    call.message.chat.id,
                    "📌 عضویت اجباری:\n"
                    "یک خط بفرستید به شکل زیر:\n"
                    "<chat_id> <invite_url اختیاری>\n"
                    "مثال:\n"
                    "-1001234567890 https://t.me/+xxxx",
                    reply_markup=types.ReplyKeyboardRemove()
                )
                bot.answer_callback_query(call.id)
                return

            if data == "A_SET_WEBTXT":
                if not self._require_admin(call.message):
                    bot.answer_callback_query(call.id); return
                self.db.set_user_state(call.from_user.id, "ADMIN_SET_WEBAPP_TEXT", {})
                bot.send_message(call.message.chat.id, "🖥 متن نمایشی در WebApp را ارسال کنید:", reply_markup=types.ReplyKeyboardRemove())
                bot.answer_callback_query(call.id)
                return

            if data == "A_SET_CAP":
                if not self._require_admin(call.message):
                    bot.answer_callback_query(call.id); return
                self.db.set_user_state(call.from_user.id, "ADMIN_SET_CAPTION", {})
                bot.send_message(call.message.chat.id, "🧩 متن ضمیمه پیام‌ها را ارسال کنید:", reply_markup=types.ReplyKeyboardRemove())
                bot.answer_callback_query(call.id)
                return

            if data == "A_NEW_COURSE":
                if not self._require_admin(call.message):
                    bot.answer_callback_query(call.id); return
                self.db.set_user_state(call.from_user.id, "ADMIN_NEW_COURSE_TITLE", {})
                bot.send_message(call.message.chat.id, "➕ عنوان دوره را بفرستید:", reply_markup=types.ReplyKeyboardRemove())
                bot.answer_callback_query(call.id)
                return

            if data == "A_LIST_COURSES":
                if not self._require_admin(call.message):
                    bot.answer_callback_query(call.id); return
                self._admin_list_courses(call.message.chat.id)
                bot.answer_callback_query(call.id)
                return

            if data.startswith("A_TCOURSE_"):
                if not self._require_admin(call.message):
                    bot.answer_callback_query(call.id); return
                cid = int(data.split("_", 2)[2])
                self.db.toggle_course(cid)
                self._admin_list_courses(call.message.chat.id)
                bot.answer_callback_query(call.id, "OK")
                return

            if data == "A_UPLOAD":
                if not self._require_admin(call.message):
                    bot.answer_callback_query(call.id); return
                self._admin_choose_course_for_upload(call.message.chat.id, call.from_user.id)
                bot.answer_callback_query(call.id)
                return

            if data.startswith("A_UC_"):
                if not self._require_admin(call.message):
                    bot.answer_callback_query(call.id); return
                cid = int(data.split("_", 2)[2])
                self.db.set_user_state(call.from_user.id, "ADMIN_UPLOAD_LESSON_TITLE", {"course_id": cid})
                bot.send_message(call.message.chat.id, "📝 عنوان درس/جلسه را بفرستید:", reply_markup=types.ReplyKeyboardRemove())
                bot.answer_callback_query(call.id)
                return

            if data == "A_STATS":
                if not self._require_admin(call.message):
                    bot.answer_callback_query(call.id); return
                self._admin_stats(call.message.chat.id)
                bot.answer_callback_query(call.id)
                return

            if data.startswith("V_OK_"):
                if not self._require_admin(call.message):
                    bot.answer_callback_query(call.id); return
                uid = int(data.split("_", 2)[2])
                self.db.verify_user(uid, True)
                bot.answer_callback_query(call.id, "تایید شد")
                bot.send_message(uid, "✅ حساب شما تایید شد. حالا می‌توانید از بخش دوره‌ها استفاده کنید.", reply_markup=make_main_menu())
                return

            if data.startswith("V_NO_"):
                if not self._require_admin(call.message):
                    bot.answer_callback_query(call.id); return
                uid = int(data.split("_", 2)[2])
                self.db.verify_user(uid, False)
                bot.answer_callback_query(call.id, "رد شد")
                bot.send_message(uid, "⛔️ تایید انجام نشد. برای پیگیری به پشتیبانی پیام بدهید.", reply_markup=make_main_menu())
                return

            bot.answer_callback_query(call.id)

    def _admin_menu_inline(self) -> types.InlineKeyboardMarkup:
        enabled = "✅ روشن" if self._bot_enabled() else "⛔️ خاموش"
        kb = glass_inline([
            [types.InlineKeyboardButton(f"🤖 وضعیت بات: {enabled}", callback_data="A_TOGGLE_BOT")],
            [types.InlineKeyboardButton("➕ ساخت دوره", callback_data="A_NEW_COURSE"),
             types.InlineKeyboardButton("📦 آپلود درس", callback_data="A_UPLOAD")],
            [types.InlineKeyboardButton("🗂 مدیریت دوره‌ها", callback_data="A_LIST_COURSES"),
             types.InlineKeyboardButton("📊 آمار کاربران", callback_data="A_STATS")],
            [types.InlineKeyboardButton("👮 افزودن ادمین", callback_data="A_ADD_ADMIN"),
             types.InlineKeyboardButton("🔐 تغییر رمز", callback_data="A_SET_PW")],
            [types.InlineKeyboardButton("🧑‍💻 تنظیم پشتیبان", callback_data="A_SET_SUPPORT"),
             types.InlineKeyboardButton("⏳ تایم لینک", callback_data="A_SET_LINKEXP")],
            [types.InlineKeyboardButton("📌 عضویت اجباری", callback_data="A_SET_REQ")],
            [types.InlineKeyboardButton("🧾 متن ضمیمه پیام‌ها", callback_data="A_SET_CAP"),
             types.InlineKeyboardButton("🖥 متن WebApp", callback_data="A_SET_WEBTXT")],
        ])
        return kb

    def _admin_menu(self) -> types.ReplyKeyboardMarkup:
        # Keep main menu; admin uses inline panel message
        return make_main_menu()

    def _notify_admins_profile(self, tg_id: int):
        u = self.db.get_user(tg_id)
        if not u:
            return
        admins = self.db.list_admins()
        phone = u["phone"] or "-"
        username = u["username"] or "-"
        text = u["profile_text"] or "(اطلاعاتی ثبت نشده)"
        msg = (
            "📩 درخواست تایید دانش‌آموز\n"
            f"🆔 ID: <code>{tg_id}</code>\n"
            f"👤 username: @{username}\n"
            f"📱 phone: <code>{phone}</code>\n\n"
            f"🧾 متن:\n{text}"
        )
        kb = glass_inline([
            [types.InlineKeyboardButton("✅ تایید", callback_data=f"V_OK_{tg_id}"),
             types.InlineKeyboardButton("⛔️ رد", callback_data=f"V_NO_{tg_id}")]
        ])
        for a in admins:
            try:
                self.bot.send_message(a, msg, reply_markup=kb)
            except Exception:
                pass

    def _show_courses(self, chat_id: int, tg_id: int):
        courses = self.db.list_courses(only_active=True)
        if not courses:
            self.bot.send_message(chat_id, "📭 هنوز دوره‌ای اضافه نشده.", reply_markup=make_main_menu())
            return
        rows = []
        for c in courses:
            rows.append([types.InlineKeyboardButton(f"📘 {c['title']}", callback_data=f"C_{c['id']}")])
        self.bot.send_message(chat_id, "📚 دوره‌ها:", reply_markup=glass_inline(rows))

    def _show_lessons(self, chat_id: int, tg_id: int, course_id: int):
        lessons = self.db.list_lessons(course_id)
        if not lessons:
            self.bot.send_message(chat_id, "📭 هنوز درسی برای این دوره آپلود نشده.")
            return
        rows = []
        for l in lessons:
            st = l["status"]
            prefix = "⏳" if st != "ready" else "🎬"
            rows.append([types.InlineKeyboardButton(f"{prefix} {l['title']}", callback_data=f"L_{l['id']}")])
        self.bot.send_message(chat_id, "🎞 لیست درس‌ها:", reply_markup=glass_inline(rows))

    def _show_qualities(self, chat_id: int, tg_id: int, lesson_id: int):
        medias = self.db.list_media_for_lesson(lesson_id)
        if not medias:
            self.bot.send_message(chat_id, "⏳ این درس هنوز در حال پردازش/تبدیل است یا آماده نشده.")
            return
        rows = []
        for m in medias:
            size_mb = (int(m["size_bytes"]) / (1024*1024)) if int(m["size_bytes"]) else 0
            rows.append([types.InlineKeyboardButton(f"🎚 {m['quality']}  ({size_mb:.1f}MB)", callback_data=f"Q_{m['id']}")])
        caption = self.db.get_setting("msg_caption", "")
        self.bot.send_message(chat_id, f"کیفیت را انتخاب کنید:\n{caption}", reply_markup=glass_inline(rows))

    def _send_webapp_link(self, chat_id: int, tg_id: int, media_id: int):
        token = secrets.token_urlsafe(20)
        exp = now() + self._link_expiry()
        self.db.cleanup_tokens()
        self.db.create_token(token, tg_id, media_id, exp)

        url = f"{self.base_url}/wa?token={token}"
        kb = types.InlineKeyboardMarkup()
        kb.row(types.InlineKeyboardButton("▶️ باز شود", web_app=types.WebAppInfo(url=url)))
        self.bot.send_message(chat_id, "✅ برای مشاهده روی دکمه زیر بزنید:", reply_markup=kb)

    def _admin_list_courses(self, chat_id: int):
        courses = self.db.list_courses(only_active=False)
        if not courses:
            self.bot.send_message(chat_id, "هیچ دوره‌ای وجود ندارد.", reply_markup=self._admin_menu_inline())
            return
        rows = []
        for c in courses:
            status = "✅" if int(c["is_active"]) == 1 else "⛔️"
            rows.append([types.InlineKeyboardButton(f"{status} {c['title']} (toggle)", callback_data=f"A_TCOURSE_{c['id']}")])
        self.bot.send_message(chat_id, "🗂 مدیریت دوره‌ها:", reply_markup=glass_inline(rows))

    def _admin_choose_course_for_upload(self, chat_id: int, admin_id: int):
        courses = self.db.list_courses(only_active=False)
        if not courses:
            self.bot.send_message(chat_id, "ابتدا یک دوره بسازید.", reply_markup=self._admin_menu_inline())
            return
        rows = []
        for c in courses:
            rows.append([types.InlineKeyboardButton(f"📘 {c['title']}", callback_data=f"A_UC_{c['id']}")])
        self.bot.send_message(chat_id, "📦 انتخاب دوره برای آپلود درس:", reply_markup=glass_inline(rows))

    def _admin_stats(self, chat_id: int):
        users = self.db.list_users()
        total = len(users)
        verified = sum(1 for u in users if int(u["is_verified"]) == 1)
        has_phone = sum(1 for u in users if (u["phone"] or "").strip())
        msg = (
            "📊 آمار\n"
            f"👥 کل کاربران استارت‌کرده: {total}\n"
            f"✅ تاییدشده: {verified}\n"
            f"📱 شماره ثبت‌شده: {has_phone}\n"
        )
        self.bot.send_message(chat_id, msg, reply_markup=self._admin_menu_inline())

        # CSV export (optional)
        output = io.StringIO()
        w = csv.writer(output)
        w.writerow(["tg_id", "username", "phone", "is_verified", "msg_count", "last_seen"])
        for u in users:
            w.writerow([u["tg_id"], u["username"], u["phone"], u["is_verified"], u["msg_count"], u["last_seen"]])
        data = output.getvalue().encode("utf-8")
        self.bot.send_document(chat_id, ("users.csv", data, "text/csv"))

    def _download_telegram_file(self, file_id: str, out_path: str):
        # Use Bot API directly (reliable on shared hosting)
        token = self.bot.token
        r1 = requests.get(f"https://api.telegram.org/bot{token}/getFile", params={"file_id": file_id}, timeout=30)
        r1.raise_for_status()
        file_path = r1.json()["result"]["file_path"]
        url = f"https://api.telegram.org/file/bot{token}/{file_path}"
        with requests.get(url, stream=True, timeout=60) as r2:
            r2.raise_for_status()
            with open(out_path, "wb") as f:
                for chunk in r2.iter_content(chunk_size=1024 * 1024):
                    if chunk:
                        f.write(chunk)